<?php
// Enable error reporting for debugging
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Start output buffering to catch any unexpected output
ob_start();

// Add the missing function
function updateSerialNumberStatus($product_id, $serial_number, $status, $sale_id) {
    global $pdo;
    try {
        // First check if the serial number exists
        $checkStmt = $pdo->prepare("
            SELECT id FROM product_serial_numbers 
            WHERE product_id = ? AND serial_number = ?
        ");
        $checkStmt->execute([$product_id, $serial_number]);
        
        if ($checkStmt->fetch()) {
            // Update existing serial number
            $stmt = $pdo->prepare("
                UPDATE product_serial_numbers 
                SET status = ?, 
                    source_type = 'sale',
                    source_id = ?,
                    updated_at = NOW() 
                WHERE product_id = ? 
                AND serial_number = ?
            ");
            $params = [$status, $sale_id, $product_id, $serial_number];
            error_log("Updating serial number: SQL = " . $stmt->queryString . ", Params = " . json_encode($params));
            $result = $stmt->execute($params);
            error_log("Update result: " . ($result ? 'Success' : 'Failed'));
            return $result;
        } else {
            // Insert new serial number
            $stmt = $pdo->prepare("
                INSERT INTO product_serial_numbers 
                (product_id, serial_number, status, source_type, source_id, created_at, updated_at)
                VALUES (?, ?, ?, 'sale', ?, NOW(), NOW())
            ");
            $params = [$product_id, $serial_number, $status, $sale_id];
            error_log("Inserting serial number: SQL = " . $stmt->queryString . ", Params = " . json_encode($params));
            $result = $stmt->execute($params);
            error_log("Insert result: " . ($result ? 'Success' : 'Failed'));
            return $result;
        }
    } catch (PDOException $e) {
        error_log("Error updating serial number status: " . $e->getMessage());
        return false;
    }
}

try {
    session_start();
    
    // Add debugging for multiple payment processing
    $debugId = uniqid('sale_', true);
    error_log("=== Sale Processing Started [{$debugId}] ===");
    error_log("Request Method: " . $_SERVER['REQUEST_METHOD']);
    error_log("Content Type: " . ($_SERVER['CONTENT_TYPE'] ?? 'Not set'));
    
    // Check if config file exists
    if (!file_exists('../config/config.php')) {
        throw new Exception('Configuration file not found');
    }
    
    require_once '../config/config.php';
    require_once '../auth/auth.php';
    
    // Set JSON header
    header('Content-Type: application/json');
    
    // Check session
    if (!isset($_SESSION['user_id'])) {
        throw new Exception('Not authorized');
    }
    
    // Check database connection
    if (!isset($pdo)) {
        throw new Exception('Database connection not available');
    }
    
    // Get and validate input
    $input = file_get_contents('php://input');
    if (empty($input)) {
        throw new Exception('No data received');
    }
    
    $data = json_decode($input, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception('Invalid JSON data: ' . json_last_error_msg());
    }
    
    // Validate required fields
    $cart = $data['cart'] ?? [];
    $customer = $data['customer'] ?? null;
    $shipping = floatval($data['shipping'] ?? 0);
    $total = floatval($data['total'] ?? 0);
    $paid = floatval($data['paid'] ?? 0);
    $payment_method = $data['payment_method'] ?? 'Cash';
    $balance = floatval($data['balance'] ?? 0);
    $payments = $data['payments'] ?? null;
    
    if (empty($cart)) {
        throw new Exception('Cart is empty');
    }
    
    if (!$customer) {
        throw new Exception('Customer not selected');
    }
    
    // Start transaction
    $pdo->beginTransaction();

    // If multiple payments, sum and get main method
    if ($payments && is_array($payments) && count($payments) > 0) {
        error_log("=== Processing Multiple Payments [{$debugId}] ===");
        error_log("Payments count: " . count($payments));
        
        $paid = 0;
        $main_method = $payments[0]['method'];
        $creditAmount = 0;
        
        foreach ($payments as $p) {
            $amount = floatval($p['amount']);
            $paid += $amount;
            
            error_log("Payment: Method = {$p['method']}, Amount = {$amount}");
            
            // Track credit amount if this is a credit payment
            if (strtolower($p['method']) === 'credit') {
                $creditAmount = $amount;
                error_log("Credit payment detected: Amount = {$creditAmount}");
            }
        }
        // Use the first payment method as the main method
        $payment_method = strtolower($main_method);
        
        // Validate total payments match total amount
        if (abs($paid - $total) > 0.01) { // Allow for small floating point differences
            throw new Exception('Total payments must equal the total amount');
        }
        
        // Calculate actual balance for multiple payments
        $balance = $total - $paid;
    }

    // Determine payment status
    $isCredit = ($payment_method === 'credit' || $balance > 0 || ($payments && in_array('credit', array_column($payments, 'method'))));
    $status = $isCredit ? 'pending' : 'completed';
    $paymentStatus = $isCredit ? 'pending' : 'paid';
    
    // Check permission for credit sales
    if ($isCredit && !hasPermission('allow_credit_sale')) {
        throw new Exception('You do not have permission to process credit sales');
    }

    // Generate unique invoice number
    $stmt = $pdo->query("SELECT MAX(CAST(SUBSTRING(invoice_number, 4) AS UNSIGNED)) as max_num FROM sales WHERE invoice_number LIKE 'INV%'");
    $result = $stmt->fetch();
    $nextNum = ($result['max_num'] ?? 0) + 1;
    $invoiceNumber = 'INV' . str_pad($nextNum, 6, '0', STR_PAD_LEFT);

    // Insert sale
    $stmt = $pdo->prepare("INSERT INTO sales (invoice_number, customer_id, user_id, total_amount, paid_amount, payment_method, status, payment_status, shipping, cash_tendered, change_amount, created_at, updated_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW(), NOW())");
    $stmt->execute([
        $invoiceNumber,
        $customer['id'],
        $_SESSION['user_id'],
        $total,
        $paid,
        $payment_method,
        $status,
        $paymentStatus,
        $shipping,
        $data['cash_tendered'] ?? null,
        $data['change_amount'] ?? null
    ]);
    $saleId = $pdo->lastInsertId();

    // Insert sale items
    error_log("=== Inserting Sale Items [{$debugId}] ===");
    error_log("Sale ID: {$saleId}");
    error_log("Cart Items Count: " . count($cart));
    
    $stmt = $pdo->prepare("INSERT INTO sale_items (sale_id, product_id, quantity, price, total, discount, unit_price, total_price) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
    foreach ($cart as $item) {
        $itemTotal = ($item['price'] * $item['quantity']) - ($item['discount'] ?? 0);
        error_log("Inserting item: Product ID {$item['id']}, Quantity {$item['quantity']}, Price {$item['price']}, Total {$itemTotal}");
        
        $stmt->execute([
            $saleId,
            $item['id'],
            $item['quantity'],
            $item['price'],
            $itemTotal,
            $item['discount'] ?? 0,
            $item['price'],
            $itemTotal
        ]);
        
        error_log("Item inserted successfully for Product ID {$item['id']}");
        
        // Update product quantity
        $stmt2 = $pdo->prepare("UPDATE products SET quantity = quantity - ? WHERE id = ?");
        $stmt2->execute([$item['quantity'], $item['id']]);

        // Handle serial numbers - only if the product has serial numbers
        $hasSerialNumbers = !empty($item['serial_numbers']);
        
        if ($hasSerialNumbers) {
            // Product has specific serial numbers provided
            $serial_numbers = is_array($item['serial_numbers']) ? $item['serial_numbers'] : array_map('trim', explode(',', $item['serial_numbers']));
            $serial_numbers = array_filter($serial_numbers); // Remove empty values
            
            if (!empty($serial_numbers)) {
                error_log("Processing specific serial numbers for product {$item['id']}: " . implode(', ', $serial_numbers));
                
                // Determine source_type based on whether this is a credit sale
                $source_type = $isCredit ? 'credit_sale' : 'manual_adjustment';
                
                foreach ($serial_numbers as $serial) {
                    $serial = trim($serial);
                    if (!empty($serial)) {
                        $stmt3 = $pdo->prepare("
                            UPDATE product_serial_numbers 
                            SET status = 'sold',
                                source_type = ?,
                                source_id = ?,
                                updated_at = NOW()
                            WHERE product_id = ? 
                            AND serial_number = ?
                            AND status = 'available'
                        ");
                        $result = $stmt3->execute([$source_type, $saleId, $item['id'], $serial]);
                        error_log("Update result for serial {$serial}: " . ($result ? 'Success' : 'Failed'));
                    }
                }
            }
        } else {
            // Check if this product requires serial numbers but none were provided
            // This is a fallback for products that should have serial numbers but weren't scanned
            $stmt4 = $pdo->prepare("
                SELECT COUNT(*) as serial_count 
                FROM product_serial_numbers 
                WHERE product_id = ? AND status = 'available'
            ");
            $stmt4->execute([$item['id']]);
            $availableSerials = $stmt4->fetch()['serial_count'];
            
            if ($availableSerials > 0) {
                // Product has available serial numbers, update the first N available ones
                $source_type = $isCredit ? 'credit_sale' : 'manual_adjustment';
                
                $stmt5 = $pdo->prepare("
                    UPDATE product_serial_numbers 
                    SET status = 'sold',
                        source_type = ?,
                        source_id = ?,
                        updated_at = NOW() 
                    WHERE product_id = ? 
                    AND status = 'available' 
                    LIMIT ?
                ");
                $stmt5->execute([$source_type, $saleId, $item['id'], $item['quantity']]);
                error_log("Updated {$item['quantity']} available serial numbers for product {$item['id']}");
            } else {
                // Product doesn't have serial numbers, skip serial number processing
                error_log("Product {$item['id']} has no serial numbers, skipping serial number processing");
            }
        }
    }

    // Insert payments if multiple
    if ($payments && is_array($payments) && count($payments) > 0) {
        $stmt = $pdo->prepare("INSERT INTO sale_payments (sale_id, method, amount, created_at) VALUES (?, ?, ?, NOW())");
        foreach ($payments as $p) {
            $stmt->execute([$saleId, strtolower($p['method']), floatval($p['amount'])]);
        }
    } else {
        // Single payment fallback
        $stmt = $pdo->prepare("INSERT INTO sale_payments (sale_id, method, amount, created_at) VALUES (?, ?, ?, NOW())");
        $stmt->execute([$saleId, $payment_method, $paid]);
    }

    // If credit sale or has credit payment, record in credit_transactions
    if ($isCredit || $creditAmount > 0) {
        // Calculate the correct credit amount
        $creditTransactionAmount = 0;
        
        if ($creditAmount > 0) {
            // If there's a specific credit payment, use that amount
            $creditTransactionAmount = $creditAmount;
        } elseif ($balance > 0) {
            // If there's a balance remaining, use that amount
            $creditTransactionAmount = $balance;
        } else {
            // Fallback: calculate as total - paid
            $creditTransactionAmount = $total - $paid;
        }
        
        // Only create credit transaction if there's actually a credit amount
        if ($creditTransactionAmount > 0) {
            // Check if credit transaction already exists for this sale
            $checkStmt = $pdo->prepare("SELECT id FROM credit_transactions WHERE sale_id = ?");
            $checkStmt->execute([$saleId]);
            
            if (!$checkStmt->fetch()) {
                $stmt = $pdo->prepare("INSERT INTO credit_transactions (customer_id, sale_id, amount, type, status, due_date, created_at) VALUES (?, ?, ?, 'credit', 'pending', DATE_ADD(NOW(), INTERVAL 30 DAY), NOW())");
                $stmt->execute([
                    $customer['id'],
                    $saleId,
                    $creditTransactionAmount
                ]);
            }
        }
    }

    // Commit the transaction
    $commitResult = $pdo->commit();
    error_log("Transaction commit result: " . ($commitResult ? 'Success' : 'Failed'));
    error_log("=== Sale Processing Completed Successfully [{$debugId}] ===");

    // Clear any output buffer
    ob_clean();
    
    echo json_encode([
        'success' => true,
        'sale_id' => $saleId,
        'receipt_url' => '../receipts/sales_receipt.php?sale_id=' . $saleId
    ]);
    
} catch (PDOException $e) {
    if (isset($pdo) && $pdo->inTransaction()) {
        $pdo->rollBack();
    }
    ob_clean();
    error_log("Database error in process_sale.php: " . $e->getMessage());
    echo json_encode([
        'success' => false,
        'message' => 'Database error occurred while processing the sale. Please try again.',
        'debug' => $e->getMessage()
    ]);
} catch (Exception $e) {
    if (isset($pdo) && $pdo->inTransaction()) {
        $pdo->rollBack();
    }
    ob_clean();
    error_log("Error in process_sale.php: " . $e->getMessage());
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
} catch (Error $e) {
    if (isset($pdo) && $pdo->inTransaction()) {
        $pdo->rollBack();
    }
    ob_clean();
    error_log("PHP Error in process_sale.php: " . $e->getMessage());
    echo json_encode([
        'success' => false,
        'message' => 'A system error occurred',
        'debug' => $e->getMessage()
    ]);
}

// End output buffering
ob_end_flush(); 